/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   ERI Request tables
::   Copyright   :   (C)2003-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   None
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file BlackBoxFuncs.h
    \brief API definition of the black box recorder protocol */

#ifndef __BLACKBOXFUNCS_H
#define __BLACKBOXFUNCS_H

/*---- INCLUDE FILES --------------------------------------------------------------------------------------*/
#include <typedefn.h>

/*---- DEFINES --------------------------------------------------------------------------------------------*/
#define MAX_BLACKBOX_NAME_LEN            32
#define MAX_BLACKBOX_FLAG_TEXT_LEN       12
#define MAX_BLACKBOX_PERSONAL_TEXT_LEN   48

/*---- TYPEDEF --------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! Black box buffer info */ 
typedef struct  
{   
    uint2   NumElements;
    uint2   CurrentIndex;
    uint2   NumBufferRows;
    uint4   SampleRate;
} S_BlackBoxBufferInfo;

/*! Desribes the type of an element */
typedef enum
{
    ELEMENT_SINT1,
    ELEMENT_SINT2,
    ELEMENT_SINT3,
    ELEMENT_SINT4,
    ELEMENT_UINT1,
    ELEMENT_UINT2,
    ELEMENT_UINT3,
    ELEMENT_UINT4,
    ELEMENT_FLAGS
} E_ElementType;

/*! Black box buffer data entry info */ 
typedef struct  
{   
    uint4   Multiplier;                     /*!< The multipler to apply to an element to convert it into engineering units */
    uint4   Divisor;                        /*!< The divisor to apply to the data to convert it into engineering units */
    sint4   Offset;                         /*!< The offset to apply to the data */
    E_ElementType eElementType;             /*!< The element type, used when interpreting the element */
    char    szName[MAX_BLACKBOX_NAME_LEN];  /*!< The name of the element */
} S_BlackBoxBufferElementInfo;

/* Black box buffer data flag info */ 
typedef struct
{
    char    szName[MAX_BLACKBOX_NAME_LEN];
    char    szTrueTxt[MAX_BLACKBOX_FLAG_TEXT_LEN];
    char    szFalseTxt[MAX_BLACKBOX_FLAG_TEXT_LEN];
} S_BlackBoxBufferFlagInfo;

/* Black box buffer Personal info */ 
typedef struct
{
    char    szFieldText[MAX_BLACKBOX_PERSONAL_TEXT_LEN];
} S_BlackBoxPersonalInfo;

typedef enum
{
    PERSONAL_CUSTOMER,
    PERSONAL_DEALER,
    MAX_PERSONAL_TYPE
} E_PersonalType;

typedef enum
{
    PERSONAL_FIELD_NAME,
    PERSONAL_FIELD_ADDRESS1,
    PERSONAL_FIELD_ADDRESS2,
    PERSONAL_FIELD_CITY,
    PERSONAL_FIELD_ZIPCODE,
    PERSONAL_FIELD_COUNTRY,
    PERSONAL_FIELD_PHONE_NUM,
    PERSONAL_FIELD_CODE,
    MAX_PERSONAL_FIELD_TYPE
} E_PersonalFieldType;

typedef NativeBool (*GetBBBufferInfoFuncPtr)         (uint1 in_u1BufferNum, S_BlackBoxBufferInfo *out_pBuffInfo);
typedef NativeBool (*GetBBBufferElementInfoFuncPtr)  (uint1 in_u1BufferNum, uint2 in_u2ElementNum, S_BlackBoxBufferElementInfo *pBuffElementInfo);
typedef NativeBool (*GetBBBufferDataFuncPtr)         (uint1 in_u1BufferNum, uint2 in_u2ElementNum, void *out_pData, uint2 *out_pu2NumBytes);
typedef NativeBool (*GetBBBufferDataFlagInfoFuncPtr) (uint1 in_u1BufferNum, uint2 in_u2ElementNum, uint1 in_u1BitNumber, S_BlackBoxBufferFlagInfo *out_pBuffFlagInfo);
typedef NativeBool (*GetBBPersonalInfoFuncPtr)       (E_PersonalType in_ePerson, E_PersonalFieldType in_eField, S_BlackBoxPersonalInfo *out_pPersonalInfo);

/*! Use this mask to identify those attributes that are to have values different from the default. The
    existance of this mask will allow this data structure to be extended as required without impacting
    upon any existing products that are using this structure */
typedef enum
{
    USE_GET_BUFF_INFO           = 0x01, /* Set S_RegisterBlackBoxProperties::pfGetBufferInfo */
    USE_GET_BUFF_ELEMENT_INFO   = 0x02, /* Set S_RegisterBlackBoxProperties::pfGetBufferElementInfo */
    USE_GET_BUFF_DATA_FLAG      = 0x04, /* Set S_RegisterBlackBoxProperties::pfGetBufferDataFlagInfo */
    USE_GET_PERSONAL_INFO       = 0x08, /* Set S_RegisterBlackBoxProperties::pfGetPersonalInfo */
    USE_GET_BUFF_DATA           = 0x10, /* Set S_RegisterBlackBoxProperties::pfGetData */
    USE_SET_NUM_BUFFERS         = 0x20, /* Set S_RegisterBlackBoxProperties::u1NumOfBuffers */
} E_BlackBoxPropertiesMask;

/*! Data structure used to describe the attributes that are applicable to the registration of a black box capabale
    application. */
typedef struct
{
/*! Attribute mask that is used to identify those attributes that have been set. Logic-OR the appropriate mask
    of each member that has been assigned a value.
    \code
    S_RegisterBlackBoxProperties BBRegisterObj;
    ...
    // Want to provide values for the BBRegisterObj.pfGetBufferInfo and BBRegisterObj.pfGetBufferElementInfo members
    BBRegisterObj.uValidAttributesMask = GET_BUFFER_INFO | GET_BUFFER_ELEMENT_INFO;
    \endcode  */
    E_BlackBoxPropertiesMask uValidAttributesMask;
/*! Get buffer information function pointer. This function is executed when a GET_BUFFER_INFO packet is
    received. */
    GetBBBufferInfoFuncPtr pfGetBufferInfo;
/*! Get element information function pointer. This function is executed when a GET_BUFFER_ELEMENT_INFO packet
    is received. Set with the \ref USE_GET_BUFF_ELEMENT_INFO. */
    GetBBBufferElementInfoFuncPtr pfGetBufferElementInfo;
/*! Get buffer flag data function pointer. This function is executed when a GET_BUFFER_DATA_FLAG_INFO packet
    is received. Set with the \ref USE_GET_BUFF_DATA_FLAG mask. */ 
    GetBBBufferDataFlagInfoFuncPtr pfGetBufferDataFlagInfo;
/*! Get flag data function pointer. This function is executed when a GET_PERSONAL_INFO packet is
    received. Set with the \ref USE_GET_PERSONAL_INFO mask. */ 
    GetBBPersonalInfoFuncPtr pfGetPersonalInfo;
/*! Get data function pointer. Executed upon receipt of a GET_BUFFER_DATA packet. Set with the mask
    \ref USE_GET_BUFF_DATA */
    GetBBBufferDataFuncPtr pfGetData;
/*! The number of buffers supported by the module. Default is zero. Set with \ref USE_SET_NUM_BUFFERS */
    uint1 u1NumOfBuffers;
} S_RegisterBlackBoxProperties;

#pragma pack()
/*---- EXTERNALS ------------------------------------------------------------------------------------------*/

/*---- PROTOTYPES -----------------------------------------------------------------------------------------*/

#endif /* __BLACKBOXFUNCS_H */

/*---- END OF FILE ----------------------------------------------------------------------------------------*/
